#!/usr/bin/env python
#
# Copyright 2012 John-Mark Gurney.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
#	$Id: //depot/python/pyfp/pyfp-0.5/encstream.py#1 $
#

'''This module implements a couple functions to encrypt and decrypt a file
object.  The keys from the provided key generator are used for encryption
and validation of both the cipher text and the plain text blocks.'''

__version__ = '''$Revision: #1 $'''
# $Id: //depot/python/pyfp/pyfp-0.5/encstream.py#1 $

import aes
import array
import hashlib
import hmac
import random

def _fromstr(x):
	return int(x.encode('hex'), 16)

def _makestr(x, cnt):
	s = '%0*x' % (cnt * 2, x)
	return s.decode('hex')

def _geniv(cnt):
	return _makestr(random.getrandbits(cnt * 8), cnt)

# 64k * 16 will be 1MB
def encfile(fp, cipher, cipherkeygen, hmackeygen=None, blksize=1024):
	'''Encrypt fp using cipher and keys provided by the iter keygen.
	blksize is measured in cipher blocks.

	This is a generator that yields strings.  The simpliest way to use
	the function is cipherout.writelines(encfile(fp, cipher, keygen)).'''

	cbs = cipher.blockSize
	tblksize = blksize * cbs

	# Make sure we have enough space for our block count + padding bit
	assert cbs > 4

	if hmackeygen is None:
		hmackeygen = cipherkeygen

	msghmac = hmac.new(hmackeygen.next(), digestmod=hashlib.sha256)

	exit = False
	while not exit:
		blk = fp.read(tblksize)
		if not blk:
			exit = True

		msghmac.update(blk)

		headerhmac = hmac.new(hmackeygen.next(),
		    digestmod=hashlib.sha256)

		ci = cipher(cipherkeygen.next())

		cipherhmac = hmac.new(hmackeygen.next(),
		    digestmod=hashlib.sha256)

		iv = _geniv(cbs)

		cbc = aes.CipherModes('CBC', ci, iv)

		if len(blk) % cbs == 0:
			padded = False
		else:
			padded = True
			blk = aes.append_PKCS7_padding(blk)

		blkcnt = len(blk) / cbs
		info = (padded << (cbs * 8 - 1)) | blkcnt
		#print 'i:', hex(info)
		info = _makestr(info, cbs)

		header = iv + ''.join(chr(x) for x in cbc.encrypt(info))
		headerhmac.update(header)

		ciphertext = cbc.encrypt(blk).tostring()
		cipherhmac.update(ciphertext)


		#print 'hhmd:', `headerhmac.digest()`
		yield headerhmac.digest()
		#print 'header:', `header`
		yield header

		if exit:
			assert exit and blkcnt == 0 and not ciphertext

		if ciphertext:
			#print 'chmd:', `cipherhmac.digest()`
			yield cipherhmac.digest()
			#print 'ciphertext:', len(ciphertext)
			yield ciphertext

	#print 'mhmd:', `msghmac.digest()`
	yield msghmac.digest()

def decfile(fp, cipher, cipherkeygen, hmackeygen=None):
	'''Decrypt data from fp using cipher and keys provided by the iter
	keygen.
	blksize is measured in cipher blocks.

	This is a generator that yields plain text strings.  The simpliest
	way to use the function is
	plaintextfp.writelines(encfile(fp, cipher, keygen)).'''

	cbs = cipher.blockSize
	maxinfo = None

	if hmackeygen is None:
		hmackeygen = cipherkeygen

	msghmac = hmac.new(hmackeygen.next(), digestmod=hashlib.sha256)

	while True:
		headerhmac = hmac.new(hmackeygen.next(),
		    digestmod=hashlib.sha256)

		ci = cipher(cipherkeygen.next())

		cipherhmac = hmac.new(hmackeygen.next(),
		    digestmod=hashlib.sha256)

		headerhmacdata = fp.read(headerhmac.digest_size)
		#print 'hhmd:', `headerhmacdata`
		if not headerhmacdata:
			return

		header = fp.read(cbs * 2)
		#print 'header:', `header`
		iv = header[:cbs]
		info = header[cbs:]

		headerhmac.update(header)
		if headerhmacdata != headerhmac.digest():
			raise RuntimeError('Header HMAC not correct')

		cbc = aes.CipherModes('CBC', ci, iv)

		info = sum(i * 256**(cbs - x - 1) for x, i in
		    enumerate(cbc.decrypt(info)))
		#print 'pi:', hex(info)
		padded = bool(info & (1 << (cbs * 8 - 1)))
		info &= (1 << (cbs * 8 - 1)) - 1

		if info == 0:
			break

		if maxinfo is None:
			maxinfo = info

		if info > maxinfo:
			raise RuntimeError('ERROR: info increased!')
		#print padded, info
		cipherhmacdata = fp.read(cipherhmac.digest_size)
		#print 'chmd:', `cipherhmacdata`
		tmp = fp.read(cbs * info)
		cipherhmac.update(tmp)

		if cipherhmacdata != cipherhmac.digest():
			raise RuntimeError('Cipher HMAC not correct')

		#print 'tmp:', `tmp[:64]`
		blk = cbc.decrypt(tmp).tostring()
		#print 'blk:', `blk[:64]`
		if padded:
			#print 'stripping:', `blk`
			blk = aes.strip_PKCS7_padding(blk)
		#print 'l:', len(blk)
		msghmac.update(blk)
		yield blk

	msghmacdata = fp.read(msghmac.digest_size)
	#print 'mhmd:', `msghmacdata`
	if msghmacdata != msghmac.digest():
		raise RuntimeError('Body HMAC not correct')
